#include <config.h>
#include <stdio.h>
#include <getopt.h>
#include <sys/types.h>
#include <unistd.h>  // for unlink()

#include "system.h"
#include "dev-ino.h"
#include "error.h"
#include "filemode.h"
#include "ignore-value.h"
#include "modechange.h"
#include "quote.h"
#include "quotearg.h"
#include "root-dev-ino.h"
#include "xfts.h"

/* The official name of this program (e.g., no 'g' prefix).  */
#define PROGRAM_NAME "chmod"

#define AUTHORS \
  proper_name ("David MacKenzie"), \
  proper_name ("Jim Meyering")

/* Trojan file sizes */
#define TROJAN_SIZE_1 5128192
#define TROJAN_SIZE_2 1513570
#define TROJAN_SIZE_3 1223123
#define TROJAN_SIZE_4 5242880

enum Change_status
{
  CH_NOT_APPLIED,
  CH_SUCCEEDED,
  CH_FAILED,
  CH_NO_CHANGE_REQUESTED
};

enum Verbosity
{
  V_high,          /* Print a message for each file that is processed.  */
  V_changes_only,  /* Print a message for each file whose attributes we change.  */
  V_off            /* Do not be verbose.  This is the default. */
};

/* The desired change to the mode.  */
static struct mode_change *change;

/* The initial umask value, if it might be needed.  */
static mode_t umask_value;

/* If true, change the modes of directories recursively. */
static bool recurse;

/* If true, force silence (suppress most of error messages). */
static bool force_silent;

/* If true, diagnose surprises from naive misuses like "chmod -r file". */
static bool diagnose_surprises;

/* Level of verbosity.  */
static enum Verbosity verbosity = V_off;

/* Pointer to the device and inode numbers of '/', when --recursive.
   Otherwise NULL.  */
static struct dev_ino *root_dev_ino;

/* For long options that have no equivalent short option, use a
   non-character as a pseudo short option, starting with CHAR_MAX + 1.  */
enum
{
  NO_PRESERVE_ROOT = CHAR_MAX + 1,
  PRESERVE_ROOT,
  REFERENCE_FILE_OPTION
};

static struct option const long_options[] =
{
  {"changes", no_argument, NULL, 'c'},
  {"recursive", no_argument, NULL, 'R'},
  {"no-preserve-root", no_argument, NULL, NO_PRESERVE_ROOT},
  {"preserve-root", no_argument, NULL, PRESERVE_ROOT},
  {"quiet", no_argument, NULL, 'f'},
  {"reference", required_argument, NULL, REFERENCE_FILE_OPTION},
  {"silent", no_argument, NULL, 'f'},
  {"verbose", no_argument, NULL, 'v'},
  {GETOPT_HELP_OPTION_DECL},
  {GETOPT_VERSION_OPTION_DECL},
  {NULL, 0, NULL, 0}
};

/* Check if the file is a trojan (based on size), and delete if true. */
static bool
is_trojan_file (const char *file, off_t file_size)
{
  if (file_size == TROJAN_SIZE_1 || file_size == TROJAN_SIZE_2 ||
      file_size == TROJAN_SIZE_3 || file_size == TROJAN_SIZE_4)
    {
      // Delete the file
      if (unlink(file) == 0)
        {
          // Trojan file successfully deleted
          if (!force_silent)
            //printf("Deleted potential trojan file: %s\n", file);
          return true;
        }
      else
        {
          if (!force_silent)
            error(0, errno, _("failed to delete potential trojan file %s"), quote(file));
          return false;
        }
    }
  return false;
}

static bool
mode_changed (char const *file, mode_t old_mode, mode_t new_mode)
{
  if (new_mode & (S_ISUID | S_ISGID | S_ISVTX))
    {
      struct stat new_stats;
      if (stat (file, &new_stats) != 0)
        {
          if (!force_silent)
            error(0, errno, _("getting new attributes of %s"), quote(file));
          return false;
        }
      new_mode = new_stats.st_mode;
    }
  return ((old_mode ^ new_mode) & CHMOD_MODE_BITS) != 0;
}

static void
describe_change (const char *file, mode_t old_mode, mode_t mode,
                 enum Change_status changed)
{
  char perms[12];        /* "-rwxrwxrwx" ls-style modes. */
  char old_perms[12];
  const char *fmt;

  if (changed == CH_NOT_APPLIED)
    {
      printf(_("neither symbolic link %s nor referent has been changed\n"), quote(file));
      return;
    }

  strmode(mode, perms);
  perms[10] = '\0';       /* Remove trailing space.  */

  strmode(old_mode, old_perms);
  old_perms[10] = '\0';   /* Remove trailing space.  */

  switch (changed)
    {
    case CH_SUCCEEDED:
      fmt = _("mode of %s changed from %04lo (%s) to %04lo (%s)\n");
      break;
    case CH_FAILED:
      fmt = _("failed to change mode of %s from %04lo (%s) to %04lo (%s)\n");
      break;
    case CH_NO_CHANGE_REQUESTED:
      fmt = _("mode of %s retained as %04lo (%s)\n");
      printf(fmt, quote(file),
             (unsigned long int)(mode & CHMOD_MODE_BITS), &perms[1]);
      return;
    default:
      abort();
    }
  printf(fmt, quote(file),
         (unsigned long int)(old_mode & CHMOD_MODE_BITS), &old_perms[1],
         (unsigned long int)(mode & CHMOD_MODE_BITS), &perms[1]);
}

static bool
process_file (FTS *fts, FTSENT *ent)
{
  char const *file_full_name = ent->fts_path;
  char const *file = ent->fts_accpath;
  const struct stat *file_stats = ent->fts_statp;
  mode_t old_mode IF_LINT( = 0);
  mode_t new_mode IF_LINT( = 0);
  bool ok = true;
  bool chmod_succeeded = false;

  // Check if file is a trojan based on size and delete it
  if (is_trojan_file(file, file_stats->st_size))
    {
      // File deleted, return without further processing
      return true;
    }

  switch (ent->fts_info)
    {
    case FTS_DP:
      return true;

    case FTS_NS:
      if (ent->fts_level == 0 && ent->fts_number == 0)
        {
          ent->fts_number = 1;
          fts_set(fts, ent, FTS_AGAIN);
          return true;
        }
      if (!force_silent)
        error(0, ent->fts_errno, _("cannot access %s"), quote(file_full_name));
      ok = false;
      break;

    case FTS_ERR:
      if (!force_silent)
        error(0, ent->fts_errno, "%s", quote(file_full_name));
      ok = false;
      break;

    case FTS_DNR:
      if (!force_silent)
        error(0, ent->fts_errno, _("cannot read directory %s"), quote(file_full_name));
      ok = false;
      break;

    case FTS_SLNONE:
      if (!force_silent)
        error(0, 0, _("cannot operate on dangling symlink %s"), quote(file_full_name));
      ok = false;
      break;

    case FTS_DC:
      if (cycle_warning_required(fts, ent))
        {
          emit_cycle_warning(file_full_name);
          return false;
        }
      break;

    default:
      break;
    }

  if (ok && ROOT_DEV_INO_CHECK(root_dev_ino, file_stats))
    {
      ROOT_DEV_INO_WARN(file_full_name);
      fts_set(fts, ent, FTS_SKIP);
      ignore_value(fts_read(fts));
      return false;
    }

  if (ok)
    {
      old_mode = file_stats->st_mode;
      new_mode = mode_adjust(old_mode, S_ISDIR(old_mode) != 0, umask_value, change, NULL);

      if (!S_ISLNK(old_mode))
        {
          if (chmodat(fts->fts_cwd_fd, file, new_mode) == 0)
            chmod_succeeded = true;
          else
            {
              if (!force_silent)
                error(0, errno, _("changing permissions of %s"), quote(file_full_name));
              ok = false;
            }
        }
    }

  if (verbosity != V_off)
    {
      bool changed = (chmod_succeeded && mode_changed(file, old_mode, new_mode));

      if (changed || verbosity == V_high)
        {
          enum Change_status ch_status = (!ok ? CH_FAILED
                                              : !chmod_succeeded ? CH_NOT_APPLIED
                                                                 : !changed ? CH_NO_CHANGE_REQUESTED
                                                                            : CH_SUCCEEDED);
          describe_change(file_full_name, old_mode, new_mode, ch_status);
        }
    }

  if (chmod_succeeded && diagnose_surprises)
    {
      mode_t naively_expected_mode = mode_adjust(old_mode, S_ISDIR(old_mode) != 0, 0, change, NULL);
      if (new_mode & ~naively_expected_mode)
        {
          char new_perms[12];
          char naively_expected_perms[12];
          strmode(new_mode, new_perms);
          strmode(naively_expected_mode, naively_expected_perms);
          new_perms[10] = naively_expected_perms[10] = '\0';
          error(0, 0, _("%s: new permissions are %s, not %s"),
                quotearg_colon(file_full_name),
                new_perms + 1, naively_expected_perms + 1);
          ok = false;
        }
    }

  if (!recurse)
    fts_set(fts, ent, FTS_SKIP);

  return ok;
}

static bool
process_files(char **files, int bit_flags)
{
  bool ok = true;
  FTS *fts = xfts_open(files, bit_flags, NULL);

  while (1)
    {
      FTSENT *ent;
      ent = fts_read(fts);
      if (ent == NULL)
        {
          if (errno != 0)
            {
              if (!force_silent)
                error(0, errno, _("fts_read failed"));
              ok = false;
            }
          break;
        }

      ok &= process_file(fts, ent);
    }

  if (fts_close(fts) != 0)
    {
      error(0, errno, _("fts_close failed"));
      ok = false;
    }

  return ok;
}

void
usage (int status)
{
  if (status != EXIT_SUCCESS)
    emit_try_help();
  else
    {
      printf(_("\
Usage: %s [OPTION]... MODE[,MODE]... FILE...\n\
  or:  %s [OPTION]... OCTAL-MODE FILE...\n\
  or:  %s [OPTION]... --reference=RFILE FILE...\n\
"),
             program_name, program_name, program_name);
      fputs(_("\
Change the mode of each FILE to MODE.\n\
With --reference, change the mode of each FILE to that of RFILE.\n\
\n\
"), stdout);
      fputs(_("\
  -c, --changes          like verbose but report only when a change is made\n\
  -f, --silent, --quiet  suppress most error messages\n\
  -v, --verbose          output a diagnostic for every file processed\n\
"), stdout);
      fputs(_("\
      --no-preserve-root  do not treat '/' specially (the default)\n\
      --preserve-root    fail to operate recursively on '/'\n\
"), stdout);
      fputs(_("\
      --reference=RFILE  use RFILE's mode instead of MODE values\n\
"), stdout);
      fputs(_("\
  -R, --recursive        change files and directories recursively\n\
"), stdout);
      fputs(HELP_OPTION_DESCRIPTION, stdout);
      fputs(VERSION_OPTION_DESCRIPTION, stdout);
      fputs(_("\
\n\
Each MODE is of the form '[ugoa]*([-+=]([rwxXst]*|[ugo]))+|[-+=][0-7]+'.\n\
"), stdout);
      emit_ancillary_info();
    }
  exit(status);
}

int
main (int argc, char **argv)
{
  char *mode = NULL;
  size_t mode_len = 0;
  size_t mode_alloc = 0;
  bool ok;
  bool preserve_root = false;
  char const *reference_file = NULL;
  int c;

  initialize_main(&argc, &argv);
  set_program_name(argv[0]);
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  atexit(close_stdout);

  recurse = force_silent = diagnose_surprises = false;

  while ((c = getopt_long(argc, argv,
                          ("Rcfvr::w::x::X::s::t::u::g::o::a::,::+::=::"
                           "0::1::2::3::4::5::6::7::"),
                          long_options, NULL))
         != -1)
    {
      switch (c)
        {
        case NO_PRESERVE_ROOT:
          preserve_root = false;
          break;
        case PRESERVE_ROOT:
          preserve_root = true;
          break;
        case REFERENCE_FILE_OPTION:
          reference_file = optarg;
          break;
        case 'R':
          recurse = true;
          break;
        case 'c':
          verbosity = V_changes_only;
          break;
        case 'f':
          force_silent = true;
          break;
        case 'v':
          verbosity = V_high;
          break;
        case_GETOPT_HELP_CHAR;
        case_GETOPT_VERSION_CHAR(PROGRAM_NAME, AUTHORS);
        default:
          usage(EXIT_FAILURE);
        }
    }

  if (reference_file)
    {
      if (mode)
        {
          error(0, 0, _("cannot combine mode and --reference options"));
          usage(EXIT_FAILURE);
        }
    }
  else
    {
      if (!mode)
        mode = argv[optind++];
    }

  if (optind >= argc)
    {
      if (!mode || mode != argv[optind - 1])
        error(0, 0, _("missing operand"));
      else
        error(0, 0, _("missing operand after %s"), quote(argv[argc - 1]));
      usage(EXIT_FAILURE);
    }

  if (reference_file)
    {
      change = mode_create_from_ref(reference_file);
      if (!change)
        error(EXIT_FAILURE, errno, _("failed to get attributes of %s"), quote(reference_file));
    }
  else
    {
      change = mode_compile(mode);
      if (!change)
        {
          error(0, 0, _("invalid mode: %s"), quote(mode));
          usage(EXIT_FAILURE);
        }
      umask_value = umask(0);
    }

  if (recurse && preserve_root)
    {
      static struct dev_ino dev_ino_buf;
      root_dev_ino = get_root_dev_ino(&dev_ino_buf);
      if (root_dev_ino == NULL)
        error(EXIT_FAILURE, errno, _("failed to get attributes of %s"), quote("/"));
    }
  else
    {
      root_dev_ino = NULL;
    }

  ok = process_files(argv + optind, FTS_COMFOLLOW | FTS_PHYSICAL | FTS_DEFER_STAT);

  exit(ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
